using System;
using System.Drawing;
using System.Collections;
using System.Reflection;

using Microsoft.DirectX;

using DarkStrideToolbox;


namespace DarkStride.StellarLanes.SharedDLL
{
	public class AIEntity : ComplexEntity
	{
		#region Properties
		private bool m_bThrusting = false;
		private Vector2 m_vSize = new Vector2( 10,10 );		
		private Entity m_oCurrentTarget = null;
		private double m_nTimeSinceLastTargetSearch = 0;
		private double m_nTimeSinceDeath = 0;
		private double m_nTimeSinceLastShot = 0;
		private string m_sThrustPMGUID = "";
		private string m_sShipTexture = "";
		private bool m_bUseThrustPS = true;

		private double m_nTimeBetweenShots = 0;
		private double m_nMaxAngularVel = 0;
		private double m_nMaxVelocity = 1;

		private double m_nTurning = 0;
		#endregion


		public AIEntity()
		{
			base.StructurePoints = 100;
			base.MaxStructurePoints = 100;
			base.FixedDamageRedux = 10;
			base.PercentDamageRedux = .5;

			this.LootID = LootConstants.m_cSHAGOT;

			m_nTimeBetweenShots = 3 + DSMisc.GetRnd() * 2 - 1;
			m_nMaxAngularVel = 4 + DSMisc.GetRnd() * 2 - 1;
			m_nMaxVelocity = 130 + DSMisc.GetRnd() * 160 - 80;

			base.AddChassis( new AIChassis( this ) );
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
            base.Advance(oSession, nElapsedTime);

			m_nTimeSinceLastShot += nElapsedTime;
			m_nTimeSinceDeath += nElapsedTime;

			AdvanceThrustPM();

			//AI time!
            if (oSession.Debug_NoAI == false && base.IsDead == false)
			{
				//Advance our combat instincts!
				UpdateAITarget( oSession,nElapsedTime );

				//Move us!
				this.Location = UpdateAIMovement( this.Location,nElapsedTime );
				if( this.ServerLocation != null )
				{
					this.ServerLocation = UpdateAIMovement( this.ServerLocation,nElapsedTime );
				}
			}
		}
		private void UpdateAITarget( Session oSession,double nElapsedTime )
		{
			Entity oClosestEntity = null;
            Zone oMyZone = null;
			double nClosestDist = 0;
			double nDist = 0;


			//Make sure our target is still valid
            if (m_oCurrentTarget != null && IsTargetViable(oSession, m_oCurrentTarget) == false)
			{
				m_oCurrentTarget = null;
			}

			m_nTimeSinceLastTargetSearch += nElapsedTime;

			//Find us a target!
			if( m_oCurrentTarget == null || m_nTimeSinceLastTargetSearch > 5 || m_oCurrentTarget.IsDead == true )
			{
                oMyZone = oSession.Zones[this.MostUpToDateLoc.ZoneID.ToString()];
				m_nTimeSinceLastTargetSearch = 0;

                foreach( Entity oLoopEntity in oMyZone.Entitys.Values )
				{
                    if (IsTargetViable(oSession, oLoopEntity) == true && oLoopEntity.GetType().BaseType == typeof(Entity) &&
                        oLoopEntity.MostUpToDateLoc.ZoneID == this.MostUpToDateLoc.ZoneID )
					{
						nDist = DSMath.Distance( oLoopEntity.MostUpToDateLoc.Pos.X,oLoopEntity.MostUpToDateLoc.Pos.Y,
												 this.MostUpToDateLoc.Pos.X,this.MostUpToDateLoc.Pos.Y );
						if( nDist < 4000 &&
							(
								oClosestEntity == null ||
								nDist < nClosestDist 
							)
						  )
						{
							nClosestDist = nDist;
							oClosestEntity = oLoopEntity;
						}
					}
				}

				//Do we have a target?  Kewl... or not.  Either way were done for this loop.
				m_oCurrentTarget = oClosestEntity;
			}
		}
		private bool IsTargetViable( Session oSession,Entity oEntity )
		{
            Zone oMyZone = oSession.Zones[this.MostUpToDateLoc.ZoneID.ToString()];

			return( oEntity != null &&
				oEntity.OwnerSocketID != MiscConstants.m_cNOSOCKETIDASSIGNED &&
				oEntity.IsDead == false &&
				oEntity.IsInPlay == true &&
				oEntity.IsVisible == true &&
                oEntity.MostUpToDateLoc.ZoneID == this.MostUpToDateLoc.ZoneID &&
                oMyZone.Entitys.ContainsKey(oEntity.PKey) == true );
		}
		private Location UpdateAIMovement( Location oLocationToMove,double nElapsedTime )
		{
			Location oNewLocation = new Location( oLocationToMove );
			DSMath.enumSHORTDISTTOANGLE nShortestDist = DSMath.enumSHORTDISTTOANGLE.enumSame;
			double nDiff = 0;


			if( m_oCurrentTarget == null )
			{
				//Slow us down while we hunt
				//oNewLocation.MaxVelocity = m_nMaxVelocity / 2.0;
			}
			else
			{
				//Speed us up while we chase
				//oNewLocation.MaxVelocity = m_nMaxVelocity;

				//Turn towords the prey
				nDiff = GetAngleDiffToPrey( ref nShortestDist );

				//CW
				if( nShortestDist == DSMath.enumSHORTDISTTOANGLE.enumCW )// DSMath.ShortestDirectionToRadAngle( nAdjustShipAngle,nAngle ) == DSMath.enumSHORTDISTTOANGLE.enumCW )
				{
					m_nTurning = Math.Min( 2 + nDiff,m_nMaxAngularVel );
					//oNewLocation.AngularMomentum = Math.Min( 2 + nDiff,m_nMaxAngularVel );
				}
				//CCW
				else
				{
					m_nTurning = -Math.Min( 2 + nDiff,m_nMaxAngularVel );
					//oNewLocation.AngularMomentum = -Math.Min( 2 + nDiff,m_nMaxAngularVel );
				}
			
				if( nDiff < .2 )
				{
					oNewLocation.AngularMomentum = 0;
				}

				m_bThrusting = ( nDiff < Math.PI / 4.0 );
			}


			return( oNewLocation );
		}

		public double GetAngleDiffToPrey()
		{
			DSMath.enumSHORTDISTTOANGLE nShortestDist = DSMath.enumSHORTDISTTOANGLE.enumSame;

			return( GetAngleDiffToPrey( ref nShortestDist ) );
		}
		public double GetAngleDiffToPrey( ref DSMath.enumSHORTDISTTOANGLE nShortestDist )
		{
			double nAdjustShipAngle = 0;
			double nAngle = 0;
			double nDiff = 0;


			if( m_oCurrentTarget != null )
			{
				//Turn towords the prey
				nAngle = DSMath.CalculateRadAngle( 
								this.MostUpToDateLoc.Pos.X,
								-this.MostUpToDateLoc.Pos.Y,
								m_oCurrentTarget.MostUpToDateLoc.Pos.X,
								-m_oCurrentTarget.MostUpToDateLoc.Pos.Y );

				//Get the ships angle in normal angle coordinates
				nAdjustShipAngle = Math.PI * 2 - this.MostUpToDateLoc.Angle;

				//Adjust for the fact that at angle 0, the ship is facing up
				nAngle -= Math.PI / 2.0;
				nDiff = DSMath.AngleRadDiff( nAdjustShipAngle,nAngle );

				nShortestDist = DSMath.ShortestDirectionToRadAngle( nAdjustShipAngle,nAngle );
			}


			return( nDiff );
		}

		private void AdvanceThrustPM()
		{
			Vector2 vWorldPtUpperLeftModuleCorner = Vector2.Empty;
			Vector2 vWorldPos = Vector2.Empty;
			Vector2 vRotatedWorldPos = Vector2.Empty;
			DSParticleManager oPMan = null;
			double nAngle = 0;


			if( this.IsDead == false && m_bUseThrustPS == true && 
				base.StructurePoints > 0 && Globals.Inst().IAmTheServer == false )
			{
				if( m_bThrusting == true )
				{
					if( m_sThrustPMGUID.Length == 0 )
					{
                        oPMan = SLEntityEffect.StartAIThrusterThrust(new Vector2(0, 0), new Vector2(0, 0), this.Location.ZoneID.ToString());
						m_sThrustPMGUID = oPMan.GUID;
					}
					else
					{
						oPMan = Globals.Inst().GameEngine.ParticleSystem.GetParticleManager( m_sThrustPMGUID );
					}

					//Set our emmiter position
					if( oPMan != null )
					{
						oPMan.RunDown = false;

						nAngle = this.Angle + Math.PI / 2.0;
						vRotatedWorldPos = new Vector2( 
										(float)( this.Pos.X + Math.Cos( nAngle ) * ( m_vSize.Y / 2.0 ) ),
										(float)( this.Pos.Y + Math.Sin( nAngle ) * ( m_vSize.Y / 2.0 ) ) );

						oPMan.EmitterPosition = new Vector2( vRotatedWorldPos.X,vRotatedWorldPos.Y );
						oPMan.EmitterDirection = new Vector2( (float)Math.Cos( nAngle ),(float)-Math.Sin( nAngle ) );
						oPMan.EmittersVelocity = new Vector2(	oPMan.EmitterDirection.X / 2.0f,
																oPMan.EmitterDirection.Y / 2.0f );
						oPMan.EmitVelocity = 5;
					}
					else 
					{
						TurnOffThrustPM();
					}
				}
				else if( m_sThrustPMGUID.Length > 0 )
				{
					TurnOffThrustPM();
				}
			}
			else if( m_sThrustPMGUID.Length > 0 )
			{
				TurnOffThrustPM();
			}
		}
		private void TurnOffThrustPM()
		{
			DSParticleManager oPMan = null;


			if( m_sThrustPMGUID.Length > 0 && 
				Globals.Inst().GameEngine != null )
			{
				oPMan = Globals.Inst().GameEngine.ParticleSystem.GetParticleManager( m_sThrustPMGUID );
				if( oPMan != null )
				{
					oPMan.RunDown = true;
				}
				m_sThrustPMGUID = "";
			}
		}

        public override double DealDamage(Session oSession, Entity oEntityDealingDamage, double nAmount, Region oDamageWhere)
		{
			double nDamageLeftOver = 0;
			bool bWasDeadBeforeHit = base.IsDead;


            nDamageLeftOver = base.DealDamage(oSession, oEntityDealingDamage, nAmount, oDamageWhere);

			//Cleanup
			if( base.IsDead == true && bWasDeadBeforeHit == false )
			{
				m_bThrusting = false;
				m_nTimeSinceDeath = 0;

				//Give me some random rotation as well
				if( this.AngularMomentum == 0 )
				{
					this.AngularMomentum = DSMisc.GetRnd() * 2 - 1;
				}				

				TurnOffThrustPM();
			}


			return( nDamageLeftOver );
		}
		public override void Render( RenderSettings oRenderSettings )
		{
			Vector2 vUpperLeftScreenPt = Vector2.Empty;
			Vector2 vLowerRightScreenPt = Vector2.Empty;
			System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;
			int nColor = 0;
			double nPercTrans = 0;


			if( oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor)
			{
				vUpperLeftScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,
						new Vector2( (int)( base.Pos.X - m_vSize.X / 2.0 ),
									 (int)( base.Pos.Y - m_vSize.Y / 2.0 ) ) );
				vLowerRightScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,
						new Vector2( (int)( base.Pos.X + m_vSize.X / 2.0 ),
									 (int)( base.Pos.Y + m_vSize.Y / 2.0 ) ) );

				oRenderRect = new System.Drawing.Rectangle( (int)vUpperLeftScreenPt.X,(int)vUpperLeftScreenPt.Y,
					(int)( vLowerRightScreenPt.X - vUpperLeftScreenPt.X ),
					(int)( vLowerRightScreenPt.Y - vUpperLeftScreenPt.Y ) );

				if( oRenderSettings.BaseDrawColor != System.Drawing.Color.White )
				{
					nColor = oRenderSettings.BaseDrawColor.ToArgb();
				}
				else if( base.IsDead == false )
				{
					nColor = System.Drawing.Color.White.ToArgb();
				}
				else
				{
					nColor = System.Drawing.Color.Gray.ToArgb();
				}

				nPercTrans = DSMisc.Max( oRenderSettings.PercentTransparent,this.FadeoutTransparency() );

				Globals.Inst().GameEngine.RenderTexture2D( m_sShipTexture,System.Drawing.Rectangle.Empty,
					oRenderRect,
					new Vector2( (float)( oRenderRect.Width / 2.0 ),(float)( oRenderRect.Height / 2.0 ) ),
					base.Angle,nPercTrans,false,nColor );
			}

			//Render our modules
			base.Render( oRenderSettings );
		}
		public override ArrayList GetRegions()
		{
			ArrayList oRegions = new ArrayList();
		
			RegionCircle oCircle = new RegionCircle( new Vector2( base.MostUpToDateLoc.Pos.X,base.MostUpToDateLoc.Pos.Y ),m_vSize.X / 2.0f );
			oRegions.Add( oCircle );
		
			return( oRegions );
		}

		public override double CalculatePropertyTotal(enumEntProperties nPropertyToGet)
		{
			double nPropertyTotal = base.CalculatePropertyTotal( nPropertyToGet );

			if( nPropertyToGet == enumEntProperties.Movement_AngularThrust )
			{
				nPropertyTotal += m_nTurning;
			}
			else if( nPropertyToGet == enumEntProperties.Movement_LinearThrust && m_bThrusting == true )
			{
				nPropertyTotal += m_nMaxVelocity;
			}

			return( nPropertyTotal );
		}

		public void AddModuleToEntity( Type oModuleType )
		{
			Module oNewModule = null;

			oNewModule = (Module)Activator.CreateInstance( oModuleType,new object[]{ this } );
			oNewModule.CreatedAsNewItem();
			oNewModule.Chassis = base.BaseChassis;

			base.BaseChassis.AddModule( oNewModule );
		}
		public void AddModuleToEntity( int nModuleID )
		{
			GenericModule oNewModule = null;

			oNewModule = new GenericModule( this,nModuleID );
			oNewModule.CreatedAsNewItem();
			oNewModule.Chassis = base.BaseChassis;

			base.BaseChassis.AddModule( oNewModule );
		}
		public Module GetModuleByType( Type oModuleType )
		{
			Module oLoopModule = null;
			Module oRetModule = null;

			try
			{
				for( int nModuleIdx=0 ; nModuleIdx<base.BaseChassis.Modules.Count ; nModuleIdx++ )
				{
					oLoopModule = (Module)base.BaseChassis.Modules.GetByIndex( nModuleIdx );
					if( oLoopModule.GetType() == oModuleType )
					{
						oRetModule = oLoopModule;
					}
				}
			}
			catch( System.Exception oEx )
			{
				DSMisc.ShowErrors ( oEx );
			}

			return( oRetModule );
		}
		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_nTimeBetweenShots );
			oSerialize.Set( 2,m_nMaxAngularVel );
			oSerialize.Set( 3,m_nMaxVelocity );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			m_nTimeBetweenShots = oSerialize.GetDouble( 1 );
			m_nMaxAngularVel = oSerialize.GetDouble( 2 );
			m_nMaxVelocity = oSerialize.GetDouble( 3 );
		}


		#region Properties
		public string ThrustPMGUID
		{
			get
			{
				return( m_sThrustPMGUID );
			}
			set
			{
				m_sThrustPMGUID = value;
			}
		}
		public string ShipTexture
		{
			get
			{
				return( m_sShipTexture );
			}
			set
			{
				m_sShipTexture = value;
			}
		}
		public Vector2 Size
		{
			get
			{
				return( m_vSize );
			}
			set
			{
				m_vSize = value;				
			}
		}
		public double TimeBetweenShots
		{
			get
			{
				return( m_nTimeBetweenShots );
			}
			set
			{
				m_nTimeBetweenShots = value;
			}
		}
		public double MaxAngularVel
		{
			get
			{
				return( m_nMaxAngularVel );
			}
			set
			{
				m_nMaxAngularVel = value;
			}
		}
		public bool UseThrustPS
		{
			get
			{
				return( m_bUseThrustPS );
			}
			set
			{
				m_bUseThrustPS = value;
			}
		}
		public Entity CurrentTarget
		{
			get
			{
				return( m_oCurrentTarget );
			}
		}
		#endregion
	}
	public class AIChassis : Chassis
	{
		public AIChassis()
		{
			//Do not put code here
			throw new System.Exception( "Initializing Chassis with wrong constructor." );
		}
		public AIChassis( ComplexEntity oParentEntity )
		{
			base.ParentEntity = oParentEntity;
			base.CGridWidth = 3;
			base.CGridHeight = 3;
		}
		public override bool Active
		{
			get
			{
				return(false);
			}
		}

	}

	public class Shagot : AIEntity
	{
		#region Properties
		private Vector2 m_vSize = new Vector2( 40,30 );
		private double m_nTimeSinceLastShot = 0;
		#endregion

		public Shagot()
		{
			base.ShipTexture = "Shagot";
			base.Size = m_vSize;

			this.BaseChassis.StructurePoints = 100;
			this.BaseChassis.MaxStructurePoints = this.BaseChassis.StructurePoints;
			base.FixedDamageRedux = 10;
			base.PercentDamageRedux = .5;

			this.LootID = LootConstants.m_cSHAGOT;
            this.EXPValue = 75;

			base.TimeBetweenShots = 3 + DSMisc.GetRnd() * 2 - 1;
			base.MaxAngularVel = 4 + DSMisc.GetRnd() * 2 - 1;
			base.MaxVelocity = 200 + DSMisc.GetRnd() * 60 - 30;

			base.AddModuleToEntity( typeof( UrquanPlasmaLauncher ) );
			base.GetModuleByType( typeof( UrquanPlasmaLauncher ) );
			base.AddModuleToEntity( 0 );
			base.AddModuleToEntity( 1 );
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
			UrquanPlasmaLauncher oWeapon = null;
			double nDist = 0;
			double nAngleDiff = 0;


            base.Advance(oSession, nElapsedTime);

			m_nTimeSinceLastShot += nElapsedTime;
			if( m_nTimeSinceLastShot >= 1 && this.CurrentTarget != null && Globals.Inst().IAmTheServer == true )
			{
				m_nTimeSinceLastShot = 0;

				//Is our target in front of me?
				nAngleDiff = this.GetAngleDiffToPrey();
				nDist = DSMath.Distance( this.Pos.X,this.Pos.Y,this.CurrentTarget.Pos.X,this.CurrentTarget.Pos.Y );
				if( nAngleDiff < .35 && nDist < 500 )
				{
					oWeapon = (UrquanPlasmaLauncher)base.GetModuleByType( typeof( UrquanPlasmaLauncher ) );
					oWeapon.Fire(oSession);
				}
			}
		}


		#region Properties
		public override string Name
		{
			get
			{
				return( "Shagot Fighter" );
			}
		}
		#endregion
	}
	public class Vux : AIEntity
	{
		#region Properties
		private Vector2 m_vSize = new Vector2( 38,57 );
		private double m_nTimeSinceLastLimpitLaunch = 0;
		#endregion

		public Vux()
		{
			base.ShipTexture = "VuxChassis";
			base.Size = m_vSize;

			this.BaseChassis.StructurePoints = 50;
			this.BaseChassis.MaxStructurePoints = this.BaseChassis.StructurePoints;
			base.FixedDamageRedux = 5;
			base.PercentDamageRedux = .1;

			base.UseThrustPS = true;
			this.LootID = LootConstants.m_cVUX;
            this.EXPValue = 100;
            long n = this.EXPValue;

			base.TimeBetweenShots = 5 + DSMisc.GetRnd() * 2 - 1;
			base.MaxAngularVel = 4 + DSMisc.GetRnd() * 2 - 1;
			base.MaxVelocity = 100 + DSMisc.GetRnd() * 40 - 20;

			base.AddModuleToEntity( typeof( LimpitLauncher ) );
			base.AddModuleToEntity( typeof( VuxLazer ) );
			base.AddModuleToEntity( 0 );
			base.AddModuleToEntity( 1 );
			base.AddModuleToEntity( 1 );
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
			LimpitLauncher oLimpitLauncher = null;
			VuxLazer oWeapon = null;
			double nDist = 0;
			double nAngleDiff = 0;

            base.Advance(oSession, nElapsedTime);

			if( this.CurrentTarget != null && this.IsDead == false )
			{
				nDist = DSMath.Distance( this.Pos.X,this.Pos.Y,this.CurrentTarget.Pos.X,this.CurrentTarget.Pos.Y );
				nAngleDiff = this.GetAngleDiffToPrey();

				//Fire the limpit launcher
				m_nTimeSinceLastLimpitLaunch += nElapsedTime;
				if( nDist < 900 && Globals.Inst().IAmTheServer == true && m_nTimeSinceLastLimpitLaunch > 1 )
				{
					m_nTimeSinceLastLimpitLaunch = 0;
					oLimpitLauncher = (LimpitLauncher)base.GetModuleByType( typeof( LimpitLauncher ) );
					oLimpitLauncher.Fire(oSession);
				}
			
				//Fire the lazer
				oWeapon = (VuxLazer)base.GetModuleByType( typeof( VuxLazer ) );
				if( nAngleDiff < .20 && nDist < 500 )
				{					
					oWeapon.StartFiring( nElapsedTime );
				}
				else
				{
					oWeapon.StopFiring( nElapsedTime );
				}
			}
		}
        

		#region Properties
		public override string Name
		{
			get
			{
				return( "Vux Intercepter" );
			}
		}
		#endregion
	}
	public class Asteroid : Entity
	{
		#region Properties
		private const double m_cTIMETOSTARTFADINGOUT = 20.0;
		private const double m_cTIMETOFADEOUT = 8.0;

		private Vector2 m_vSize = Vector2.Empty;
		private string m_sTextureKey = "";
		private System.Drawing.Color m_nColorToAddIn = System.Drawing.Color.White;
		#endregion


		public Asteroid()
		{
			double nMag = 0;
			double nRnd = DSMisc.GetRnd();


			this.EntityType = enumEntityType.Astrological;
			m_sTextureKey = "Asteroid_" + DSMisc.GetRnd( 1,6 ).ToString();

			//250-350
			if( nRnd < .02 )
			{
                this.LootID = LootConstants.m_cASTEROID_LARGE;
				this.StructurePoints = 150;
				this.MaxStructurePoints = this.StructurePoints;
				this.PercentDamageRedux = .05;
				this.FixedDamageRedux = 3;
				m_vSize = new Vector2( (float)( 300 + DSMisc.GetRnd() * 100 - 50 ),(float)( 300 + DSMisc.GetRnd() * 100 - 50 ) );
			}
			//150-250
			else if( nRnd < .1 )
			{
                this.LootID = LootConstants.m_cASTEROID_MEDIUM;
				this.StructurePoints = 100;
				this.MaxStructurePoints = this.StructurePoints;
				this.PercentDamageRedux = .03;
				this.FixedDamageRedux = 2;
				m_vSize = new Vector2( (float)( 200 + DSMisc.GetRnd() * 100 - 50 ),(float)( 200 + DSMisc.GetRnd() * 100 - 50 ) );
			}			
			//50-150
			else if( nRnd < .25 )
			{
                this.LootID = LootConstants.m_cASTEROID_SMALL;
				this.StructurePoints = 50;
				this.MaxStructurePoints = this.StructurePoints;
				this.PercentDamageRedux = .01;
				this.FixedDamageRedux = 1;
				m_vSize = new Vector2( (float)( 100 + DSMisc.GetRnd() * 100 - 50 ),(float)( 100 + DSMisc.GetRnd() * 100 - 50 ) );
			}			
			//20-80
			else 
			{
                this.LootID = LootConstants.m_cASTEROID_SMALL;
				this.StructurePoints = 20;
				this.MaxStructurePoints = this.StructurePoints;
				this.PercentDamageRedux = 0;
				this.FixedDamageRedux = 0;
				m_vSize = new Vector2( (float)( 50 + DSMisc.GetRnd() * 80 - 40 ),(float)( 50 + DSMisc.GetRnd() * 80 - 40 ) );
			}			

			nMag = Math.Sqrt( this.Vel.Length() );
			this.AngularMomentum = DSMisc.GetRnd() * nMag * 2 - nMag;
			this.EXPValue = (long)( ( this.Size.X * this.Size.Y ) / 5000.0 );
            this.EXPValue = DSMisc.Max(1,this.EXPValue);
            this.EXPValue = DSMisc.Min(50, this.EXPValue);
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
            base.Advance(oSession, nElapsedTime);

			if( m_vSize.Length() < 55 && this.TotalElapsedTime > m_cTIMETOSTARTFADINGOUT )
			{
				this.StartFadeout( m_cTIMETOFADEOUT );
			}
		}

        public override bool Collision(Session oSession, Entity oCollidedWith, double nElapsedTime, RegionPoint oCollisionRegion)
		{
			if( oCollidedWith.SuperMass == true )
			{
                oSession.RemoveEntity(this);
			}

            return( base.Collision(oSession, oCollidedWith, nElapsedTime, oCollisionRegion));
		}
		public override double DealDamage( Session oSession,Entity oEntityDealingDamage,double nAmount,Region oDamageWhere )
		{
			long nNumToCreate = 0;
			double nRnd = 0;
			double nDamageLeftOver = 0;
			float nNewSize = 0;
			bool bWasDeadBeforeThisDamage = this.IsDead;
            Zone oMyZone = null;
			Asteroid oNewAsteroid = null;


            nDamageLeftOver = base.DealDamage(oSession, oEntityDealingDamage, nAmount, oDamageWhere);

            if (Globals.Inst().IAmTheServer == true)
            {
                oMyZone = oSession.Zones[this.MostUpToDateLoc.ZoneID.ToString()];

                //Can we split this asteroid?
                if (bWasDeadBeforeThisDamage == false && this.IsDead == true && m_vSize.Length() > 70 &&
                    Globals.Inst().IAmTheServer == true)
                {
                    //Spin off some asteroids, i used to split off many of them, but it created to many
                    nRnd = DSMisc.GetRnd();
                    if (nRnd < .02) { nNumToCreate = 3; }
                    else if (nRnd < .1) { nNumToCreate = 2; }
                    else if (nRnd < .25) { nNumToCreate = 2; }
                    else { nNumToCreate = 1; }

                    //Make them!
                    for (int i = 0; i < nNumToCreate; i++)
                    {
                        oNewAsteroid = new Asteroid();
                        oNewAsteroid.TextureKey = this.TextureKey;
                        nNewSize = (float)DSMisc.GetRnd(m_vSize.Length() / 4.0, m_vSize.Length() / 2.0);
                        oNewAsteroid.Size = new Vector2(nNewSize, nNewSize);
                        oNewAsteroid.Vel = new Vector2((float)(this.Vel.X + DSMisc.GetRnd() * 50 - 25), (float)(this.Vel.Y + DSMisc.GetRnd() * 50 - 25));
                        oNewAsteroid.AngularMomentum = this.AngularMomentum * DSMisc.GetRnd() - .5 * 4;
                        oNewAsteroid.Pos = this.Pos;
                        oNewAsteroid.Location.ZoneID = this.Location.ZoneID;
                        oNewAsteroid.EXPValue = (long)((oNewAsteroid.Size.X * oNewAsteroid.Size.Y) / 200.0);

                        if (nDamageLeftOver > 0)
                        {
                            nDamageLeftOver = oNewAsteroid.DealDamage(oSession, oEntityDealingDamage, nDamageLeftOver, oDamageWhere);
                        }
                        if (oNewAsteroid.IsDead == false)
                        {
                            oMyZone.AddEntity(oSession,oNewAsteroid, true);
                        }
                    }
                }
                //Remove the asteroid if its dead
                if (this.IsDead == true)// && Globals.Inst().IAmTheServer == true )
                {
                    oSession.RemoveEntity(this);
                }
            }

			return( nDamageLeftOver );
		}
		public override double DelayBetweenLiteSends()
		{
			return( 5 );
		}

		public override void Render( RenderSettings oRenderSettings )
		{
			Vector2 vScreenPt = Vector2.Empty;
			Vector2 vWorldUpperLeftCorner = Vector2.Empty;
			System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;
			System.Drawing.Color oColor = System.Drawing.Color.White;
			double nPercTrans = 0;


			if( oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor )
			{
				vWorldUpperLeftCorner = new Vector2((float)( this.Pos.X - m_vSize.X / 2.0 ),
					                                (float)( this.Pos.Y - m_vSize.Y / 2.0 ) );
				vScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,vWorldUpperLeftCorner );

				oRenderRect = new System.Drawing.Rectangle( 
					(int)vScreenPt.X,(int)vScreenPt.Y,
					(int)( m_vSize.X * oRenderSettings.ZoomLevel ),
					(int)( m_vSize.Y * oRenderSettings.ZoomLevel ) );


				if( m_nColorToAddIn == System.Drawing.Color.White )
				{
					oColor = oRenderSettings.BaseDrawColor;
				}
				else
				{
					oColor = m_nColorToAddIn;
				}

				nPercTrans = DSMisc.Max( oRenderSettings.PercentTransparent,this.FadeoutTransparency() );

				Globals.Inst().GameEngine.RenderTexture2D( m_sTextureKey,
					System.Drawing.Rectangle.Empty,oRenderRect,
					new Vector2( oRenderRect.Width / 2.0f,oRenderRect.Height / 2.0f ),this.Angle,
					nPercTrans,false,oColor.ToArgb() );
			}
		}
		public override ArrayList GetRegions()
		{
			ArrayList oRegions = new ArrayList();

			RegionCircle oCircle = new RegionCircle( 
				new Vector2( base.MostUpToDateLoc.Pos.X,base.MostUpToDateLoc.Pos.Y ),
				m_vSize.X / 2.0f );
			oRegions.Add( oCircle );

			return( oRegions );
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_vSize.X );
			oSerialize.Set( 2,m_vSize.Y );
			oSerialize.Set( 3,m_sTextureKey );
			oSerialize.Set( 4,m_nColorToAddIn.ToArgb() );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			float nTempX = 0, nTempY = 0;

			
			base.DeSerialize(oSession, (DSSerialize)oSerialize.Get( 0 ) );
			nTempX = oSerialize.GetFloat( 1 );
			nTempY = oSerialize.GetFloat( 2 );
			this.m_vSize = new Vector2( nTempX,nTempY );
			m_sTextureKey = oSerialize.GetString( 3 );
			this.ColorToAddIn = System.Drawing.Color.FromArgb( oSerialize.GetInt( 4 ) );
		}


		#region Properties
		public Vector2 Size
		{
			get
			{
				return( m_vSize );
			}
			set
			{
				m_vSize = value;
			}
		}
		public string TextureKey
		{
			get
			{
				return( m_sTextureKey );
			}
			set
			{
				m_sTextureKey = value;
			}
		}
		public System.Drawing.Color ColorToAddIn
		{
			get
			{
				return( m_nColorToAddIn);
			}
			set
			{
				m_nColorToAddIn = value;
			}
		}
		public override string Name
		{
			get
			{
				return( "Asteroid" );
			}
		}
		#endregion
	}
    public class RedAsteroid : Asteroid
    {
        #region Properties
        #endregion

        public RedAsteroid()
        {
            this.StructurePoints *= 5;
            this.MaxStructurePoints *= 5;
            this.PercentDamageRedux *= 2;
            this.FixedDamageRedux *= 2;
            this.Size = new Vector2(DSMisc.Max(55, this.Size.X), DSMisc.Max(55, this.Size.Y));
            this.LootID = LootConstants.m_cREDASTEROID;
            this.ColorToAddIn = System.Drawing.Color.FromArgb(255, 100, 100);
            this.EntityType = enumEntityType.Ship;
            this.EXPValue = DSMisc.Max(1, (long)((this.Size.X * this.Size.Y) / 1000.0));
        }

        #region Properties
        public override string Name
        {
            get
            {
                return ("Red Asteroid");
            }
        }
        #endregion
    }

    public class Becon : Entity
    {
        #region Member Variables
        private const double m_cBLINKSPEED = .75;

        private Vector2 m_vSize = new Vector2( 10,10 );
        private double m_nTimeTillChange = 0;
        private int m_nColor = 0;
        #endregion

        public Becon()
        {
            this.EntityType = enumEntityType.Effect;
            m_nColor = DSMisc.GetRnd(0, 3);
            m_nTimeTillChange = DSMisc.GetRnd(0, m_cBLINKSPEED);
        }
        public override void Advance(Session oSession, double nElapsedTime)
        {
 	        base.Advance(oSession, nElapsedTime);

            this.Vel = new Vector2(0, 0);

            m_nTimeTillChange += nElapsedTime;
            if (m_nTimeTillChange > m_cBLINKSPEED)
            {
                m_nTimeTillChange = 0;
                m_nColor++;
            }

            if (this.IsDead == true)
            {
                oSession.RemoveEntity(this);
            }
        }
        public override void Render(RenderSettings oRenderSettings)
        {
            string sTextureKey = string.Empty;


            base.Render(oRenderSettings);

            if( m_nColor % 4 == 1 )
            {
                sTextureKey = "Becon_Green";
            }
            else if( m_nColor % 4 == 3 )
            {
                sTextureKey = "Becon_Red";
            }
            else 
            {
                sTextureKey = "Becon_Normal";
            }

            base.RenderSimple(oRenderSettings, sTextureKey, new Vector2(10, 10));
        }
        public override ArrayList GetRegions()
        {
            ArrayList oRegions = new ArrayList();

            RegionCircle oCircle = new RegionCircle(
                new Vector2(base.MostUpToDateLoc.Pos.X, base.MostUpToDateLoc.Pos.Y),
                m_vSize.X / 2.0f);
            oRegions.Add(oCircle);

            return (oRegions);
        }

        public override double DelayBetweenLiteSends()
        {
            return (60);
        }
        public override double DelayBetweenFullSends()
        {
            return (30);
        }

        public void SetOffset(double nOffset)
        {
            m_nTimeTillChange = nOffset;
        }


        #region Properties
        #endregion
    }
}
